﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;

    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Domain.Entities;
    using Hims.Domain.Entities.Enums;
    using Hims.Shared.Library.Enums;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;

    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels;
    using Shared.UserModels.Vendors;
    using Utilities;

    /// <summary>
    /// The vendor quotation controller.
    /// </summary>
    [Authorize]
    [Route("api/vendor-quotation")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class InitialVendorQuotationController : BaseController
    {
        /// <summary>
        /// The quotation service
        /// </summary>
        private readonly IQuotationService quotationService;

        /// <inheritdoc />
        public InitialVendorQuotationController(
           IQuotationService quotationService
            )
        {
            this.quotationService = quotationService;
        }

        /// <summary>
        /// Called when add incoming quotation asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("add-incoming-quotation")]
        public async Task<ActionResult> OnAddIncomingQuotationAsync([FromBody] IncomingQuotationHeaderModel model)
        {
            model = (IncomingQuotationHeaderModel)EmptyFilter.Handler(model);
            if(model.Products.Count == 0)
            {
                return BadRequest("No Product found.");
            }

            var response = await this.quotationService.OnIncomingQuotationAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the incoming quotation asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-incoming-quotation")]
        public async Task<ActionResult> FetchIncomingQuotationAsync([FromBody] IncomingQuotationHeaderModel model)
        {
            model ??= new IncomingQuotationHeaderModel();
            var response = await this.quotationService.FetchIncomingQuotationAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Deletes the incoming quotation asynchronous.
        /// </summary>
        /// <param name="id">The identifier.</param>
        /// <returns></returns>
        [HttpGet]
        [Route("delete-quotation")]
        public async Task<ActionResult> DeleteIncomingQuotationAsync(long id)
        {
            return this.Success(await this.quotationService.DeleteQuotationAsync(id));
        }


        /// <summary>
        /// Accepts the quotation asynchronous.
        /// </summary>
        /// <param name="id">The identifier.</param>
        /// <param name="createdBy">The created by.</param>
        /// <returns></returns>
        [HttpGet]
        [Route("accept-quotation")]
        public async Task<ActionResult> AcceptQuotationAsync(long id, int createdBy)
        {
            return this.Success(await this.quotationService.OnAcceptQuotation(id, createdBy));
        }
    }
}
